#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>

#if defined(HAVE_MYSQL) && defined(HAVE_POSTGRESQL)
#error You can not define both of HAVE_MYSQL and HAVE_POSTGRESQL.
#endif

#if defined(V20)
#warning Target of Zabbix version is 2.0.x.
#elif defined(V18)
#warning Target of Zabbix version is 1.8.x.
#endif

#if !defined(V20) && !defined(V18)
#error You should define one of V20 or V18.
#endif

#if defined(HAVE_MYSQL)
#include <mysql.h>
#elif defined(HAVE_POSTGRESQL)
#include <libpq-fe.h>
#else
#error You shuold define HAVE_MYSQL or HAVE_POSTGRESQL.
#endif

typedef struct {
   const char *sql;
   const char *expect;
   const char *version;
} verinfo;

#if defined(V20)
#define MAX_VERSION "2.0.15"
verinfo verinfos[] = {
    {
        .sql = "select value_type from items where itemid=23122",
        .expect = "0",
        .version = "2.0.15",
    },
    {
        .sql = "select name from items where itemid=23211",
        .expect = "mp Tenured Gen used",
        .version = "2.0.13 or 2.0.14",
    },
    {
        .sql = "select description from items where itemid=22746",
        .expect = "The rule will discover all disk partitions matching the global regexp \"Storage devices for SNMP discovery\".\r\n\r\n{$SNMP_COMMUNITY} is a global macro.",
        .version = "2.0.11 or 2.0.12",
    },
    {
        .sql = "select description from items where itemid=10010",
        .expect = "The processor load is calculated as system CPU load divided by number of CPU cores.",
        .version = "2.0.7, 2.0.8, 2.0.9 or 2.0.10",
    },
    {
        .sql = "select value_type from items where itemid=22822 limit 1",
        .expect = "0",
        .version = "2.0.6",
    },
    {
        .sql = "select yaxisside from graphs_items where gitemid=1442 limit 1",
        .expect = "1",
        .version = "2.0.5",
    },
    {
        .sql = "select delay_flex from items where itemid=22701 limit 1",
        .expect = "",
        .version = "2.0.4",
    },
    {
        .sql = "select description from help_items where itemtype=0 and key_='vfs.file.exists[file]' limit 1",
        .expect = "Check if file exists. 0 - file does not exist, 1 - file exists",
        .version = "2.0.3",
    },
    {
        .sql = "select key_ from help_items where itemtype=0 and description='Performs a DNS query. On success returns a character string with the required type of information.'",
        .expect = "net.dns.record[&lt;ip&gt;,zone,&lt;type&gt;,&lt;timeout&gt;,&lt;count&gt;]",
        .version = "2.0.2",
    },
    {
        .sql = "select theme from users where userid=1 limit 1",
        .expect = "default",
        .version = "2.0.1",
    },
};
#elif defined(V18)
#define MAX_VERSION "1.8.22"
verinfo verinfos[] = {
    {
        .sql = "select description from help_items where itemtype=0 and  key_='vfs.file.exists[file]' limit 1",
        .expect = "Check file existence. 0 - file does not exist, 1 - file exists",
        .version = "1.8.15, 1.8.16, 1.8.17, 1.8.18, 1.8.19, 1.8.20, 1.8.21 or 1.8.22",
    },
    {
        .sql = "select formula from items where itemid=20161 limit 1",
        .expect = "8",
        .version = "1.8.13 or 1.8.14",
    },
    {
        .sql = "select dns from hosts where hostid=10008 limit 1",
        .expect = "",
        .version = "1.8.9, 1.8.10, 1.8.11 or 1.8.12",
    },
    {
        .sql = "select value_type from items where itemid=10414 limit 1",
        .expect = "3",
        .version = "1.8.8",
    },
    {
        .sql = "select description from help_items where itemtype=5 and key_='zabbix[process,&lt;type&gt;,&lt;num&gt;,&lt;state&gt;]' limit 1",
        .expect = "Time a particular Zabbix process or a group of processes (identified by &lt;type&gt; and &lt;num&gt;) spent in &lt;state&gt; in percentage.",
        .version = "1.8.6 or 1.8.7",
    },
    {
        .sql = "select description from help_items where itemtype=5 and key_='zabbix[process,&lt;type&gt;,&lt;num&gt;,&lt;state&gt;]' limit 1",
        .expect = "Time a particular Zabbix process or a group of processes (identified by <type> and <num>) spent in <state> in percentage.",
        .version = "1.8.5",
    },
    {
        .sql = "select description from help_items where itemtype=5 and key_='zabbix[boottime]' limit 1",
        .expect = "Startup time of Zabbix server, Unix timestamp.",
        .version = "1.8.4",
    },
    {
        .sql = "select host from hosts where hostid=10017 limit 1",
        .expect = "Zabbix server",
        .version = "1.8.3",
    },
    {
        .sql = "select key_ from items where itemid=17409 limit 1",
        .expect = "perf_counter[\\PhysicalDisk(_Total)\\Avg. Disk Read Queue Length]",
        .version = "1.8.2",
    },
    {
        .sql = "select users_status from usrgrp where usrgrpid=9 limit 1",
        .expect = "1",
        .version = "1.8.1",
    },
};
#else
#error You shuold define one of V20 or V18.
#endif

#if defined(HAVE_MYSQL)
int check_version_M(MYSQL *conn, verinfo s);
#elif defined(HAVE_POSTGRESQL)
int check_version_P(PGconn *pconn, verinfo s);
#endif
void usage(char *);

void usage(char *prog)
{
    fprintf(stdout, "Usage: %s [-s server] [-u dbuser] [-p password] [-d database]\n", prog);
    fprintf(stdout, "\t-s : server ip address or hostname (default: localhost).\n");
    fprintf(stdout, "\t-u : dbuser on database (default: root).\n");
    fprintf(stdout, "\t-p : dbuser's password (default: mysqlpass).\n");
    fprintf(stdout, "\t-d : Zabbix's database (default: zabbix).\n");
    return;
}
 
int main(int argc, char *argv[])
{
#if defined(HAVE_MYSQL)
    MYSQL *conn;
#elif defined(HAVE_POSTGRESQL)
    char conninfo[256];
    PGconn     *pconn;
#endif
 
    char *server = "localhost";
    char *user = "root";
    char *password = "mysqlpass";
    char *database = "zabbix";
    int sf = 0, uf = 0, pf = 0, df = 0;

    int i, opt;

    fprintf(stdout, "This version can research up to Zabbix " MAX_VERSION ".\n");

    while ((opt = getopt(argc, argv, "s:u:p:d:h")) != -1) {
        switch (opt) {
        case 's':
            if (sf) {
                usage(argv[0]);
                return 2;
            }
            sf = 1;
            server = optarg;
            break;
        case 'u':
            if (uf) {
                usage(argv[0]);
                return 2;
            }
            uf = 1;
            user = optarg;
            break;
        case 'p':
            if (pf) {
                usage(argv[0]);
                return 2;
            }
            pf = 1;
            password = optarg;
            break;
        case 'd':
            if (df) {
                usage(argv[0]);
                return 2;
            }
            df = 1;
            database = optarg;
            break;
        case 'h':
            usage(argv[0]);
            return 0;
            break;
        default: /* Never reach */
            fprintf(stderr, "Ooops! Unknown Error!\n");
            return 3;
        }
    }

#if defined(HAVE_MYSQL)
    if ((conn = mysql_init(NULL)) == NULL) {
        fprintf(stderr, "Can not initialize.\n");
        return 1;
    }
 
    if (mysql_real_connect(conn, server, user, password, database, 0, NULL, 0) == NULL) {
        fprintf(stderr, "%s\n", mysql_error(conn));
        return 1;
    }

    for (i = 0; i < sizeof(verinfos)/sizeof(verinfo); i++) {
        if (check_version_M(conn, verinfos[i]) == 0) {
            fprintf(stdout, "Your db may be built on %s\n", verinfos[i].version);
            goto finish;
        }
    }
#elif defined(HAVE_POSTGRESQL)
    if (strcmp(password, "") == 0) 
        snprintf(conninfo, 255, "user=%s dbname=%s host=%s", user, database, server);
    else
        snprintf(conninfo, 255, "user=%s dbname=%s host=%s password=%s", user, database, server, password);
    pconn = PQconnectdb(conninfo);
    if (PQstatus(pconn) != CONNECTION_OK)
    {
        fprintf(stderr, "Connection to database failed: %s", PQerrorMessage(pconn));
        PQfinish(pconn);
        return 1;
    }

    for (i = 0; i < sizeof(verinfos)/sizeof(verinfo); i++) {
        if (check_version_P(pconn, verinfos[i]) == 0) {
            fprintf(stdout, "Your db may be built on %s\n", verinfos[i].version);
            goto finish;
        }
    }
#endif

#if defined(V20)
    fprintf(stdout, "Your db may be built on 2.0.0\n");
#elif defined(V18)
    fprintf(stdout, "Your db may be built on 1.8\n");
#endif

finish:
#if defined(HAVE_MYSQL)
    mysql_close(conn);
#elif defined(HAVE_POSTGRESQL)
    PQfinish(pconn);
#endif

    return 0;
}

#if defined(HAVE_MYSQL)
int check_version_M(MYSQL *conn, verinfo s)
{
    MYSQL_ROW row;
    MYSQL_RES *res;
    int ret = 1;

    if (mysql_query(conn, s.sql)) {
        fprintf(stderr, "%s\n", mysql_error(conn));
    }

    res = mysql_use_result(conn);
    if ((row = mysql_fetch_row(res)) != NULL) {
        if (strcmp(row[0], s.expect) == 0)
            ret = 0;
    }

    mysql_free_result(res);
    return ret;
}
#elif defined(HAVE_POSTGRESQL)
int check_version_P(PGconn *conn, verinfo s)
{
    PGresult *res;
    int ret = 1;

    res = PQexec(conn, s.sql);
    if (PQresultStatus(res) != PGRES_TUPLES_OK) {
        PQclear(res);
        return 1;
    }

    if (PQntuples(res) != 0) {
        if (strcmp(PQgetvalue(res, 0, 0), s.expect) == 0)
            ret = 0;
    }

    PQclear(res);
    return ret;
}
#endif
